/*
 * File:   main_SIRCS_Sniffer.c
 * Author: witkatz
 * Created on 07. August 2014, 21:36
 *
 * Proc: 16F876
 * Compiler: X8 V1.32
 */

#define _XTAL_FREQ 4000000
#include <xc.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include "lcd.h"

// CONFIG
#pragma config FOSC = HS        // Oscillator Selection bits (HS oscillator)
#pragma config WDTE = OFF       // Watchdog Timer Enable bit (WDT disabled)
#pragma config PWRTE = OFF      // Power-up Timer Enable bit (PWRT disabled)
#pragma config CP = OFF         // FLASH Program Memory Code Protection bits (Code protection off)
#pragma config BOREN = OFF      // Brown-out Reset Enable bit (BOR disabled)
#pragma config LVP = OFF        // Low Voltage In-Circuit Serial Programming Enable bit (RB3 is digital I/O, HV on MCLR must be used for programming)
#pragma config CPD = OFF        // Data EE Memory Code Protection (Code Protection off)
#pragma config WRT = ON         // FLASH Program Memory Write Enable (Unprotected program memory may be written to by EECON control)

// Pinbelegung IR
#define IrRcv_Portbit 0b00000100
#define IrRcv_Port PORTA

// globale Variablen
volatile uint8_t cnt_100us;     // wird in Interrupt Routine alle 100us inkrementiert
volatile uint8_t cnt_10ms;      // wird in Interrupt Routine alle 10ms inkrementiert
char tmpstring[4];              // Puffer
uint8_t IrRcvPort_FF;           // Fallende Flanken
uint8_t sircs_code;             // SIRCS code, bzw. command
uint8_t sircs_address;          // SIRCS12-address, ohne SIRCS-15 und SIRCS-20 Erweiterungen
uint8_t sircs_ext3;             // SIRCS-15 3 bit Erweiterung
uint8_t sircs_ext5;             // SIRCS-20 5 bit Erweiterung
uint8_t sircs_address;          // SIRCS12-address, ohne SIRCS-15 und SIRCS-20 Erweiterungen
uint8_t sircs_cnt;              // Anzahl hintereinander gesendeter gleicher Telegramme -
                                // Zeit des Tastendrucks = 45ms * Anzahl Telegramme

// Funktionsprototypen
void initMCU(void);
void delay1usLCD(void);
void delay5msLCD(void);
void delay45msLCD(void);
void fIrPort_FF(void);
int8_t SIRCS_ReceiveBlock(uint8_t, uint8_t *);
int8_t IR_SircsRcv(void);
void SircsToLCD(void);
void SircsCmdToLCD(void);
void ErrorToLcd(int8_t);

void interrupt isr(void){
    uint8_t cnt_10ms_event;
    TMR0 += 256 - 100 + 3;
    INTCONbits.T0IF = 0;
    ++cnt_100us;    // 100us Timervariable
    ++cnt_10ms_event;

    if (cnt_10ms_event == 100){
        cnt_10ms_event = 0;
        ++cnt_10ms; // 10ms Timervariable
    }
}
void main(void){
    // Initialisierungen
    int8_t err;
    initMCU();
    InitLCD(FOUR_BIT & LINES_5X7);

    // Splash
    SetDDRamAddrLCD(0x0);
    WriteStringLCD("SIRCS_Snif V1.0 ");
    __delay_ms(1000);
    SetDDRamAddrLCD(0x40);
    WriteStringLCD("waiting for IR..");

    // Main Loop
    while(1){
        err = IR_SircsRcv();
        // Fehler ausgeben
        if (err < 0){
            cnt_10ms = 0;
            sircs_code = 0;
            ErrorToLcd(err);
        }
        // nach 100ms Sendepause Ergebnis ausgeben
        else if(cnt_10ms > 10 && err == 1){
            if (sircs_code != 0){
                SircsToLCD();
                sircs_code = 0;
            }
        }
    }
}
void initMCU(void){
    // Set POrtA to dig
    PORTA = 0;
    ADCON1 = 0x06;
    TRISA = 0b111110;

    // configure interrupt
    INTCONbits.GIE = 1;
    INTCONbits.T0IE = 1;

    // configure Timer0
    OPTION_REGbits.T0CS = 0; // TMR0 Clock Source, 0 = Internal instruction cycle clock
    OPTION_REGbits.PSA = 1; // Prescaler Assignment bit, 0 = Prescaler is assigned to WDT
    TMR0 = 0;
    return;
}
int8_t IR_SircsRcv(void){
    static uint8_t sircs_code_old;
    int8_t err;

    // Warten auf fallende Flanke am IR Eingang
    fIrPort_FF();
    if ((IrRcvPort_FF & IrRcv_Portbit) == 0)
        return 1; // keine Flanke erkannt

    // Flanke erkannt -> 3000us Startbit empfangen
    sircs_code = 0;
    sircs_address = 0;
    cnt_100us = 0;
    while(cnt_100us < 25){ // 2500us berwachung
        fIrPort_FF();
        if ((IrRcvPort_FF & IrRcv_Portbit) != 0){
            return -1; // fallende Flanke detektiert -> Fehler
        }
    }

    // 7Bits SIRCS Code Block empfangen
    if (SIRCS_ReceiveBlock(0b01000000, &sircs_code) != 0){
        return -2;
    }
    
    // Command Wiederholungen zhlen
    if (cnt_10ms >= 10){ // neues Telegramm nach 100ms Sendepause?
        sircs_cnt = 0;
    }
    else if(sircs_code == sircs_code_old){
        sircs_cnt++;
    }
    else{
        sircs_cnt = 0;
    }
    cnt_10ms = 0;
    sircs_code_old = sircs_code;

    // 5Bits SIRCS Address empfangen
    if (SIRCS_ReceiveBlock(0b00010000, &sircs_address) != 0)
        return -3;

    err = SIRCS_ReceiveBlock(0b00000100, &sircs_ext3);
    if (err == 1){ // timeout?
        sircs_ext3 = 0; // keine SIRCS-15 erweiterung empfangen
        sircs_ext5 = 0;
        return 0;
    } else if (err < 0) { // Fehler?
        sircs_ext3 = 0;
        sircs_ext5 = 0;
        return -4;
    }

    err = SIRCS_ReceiveBlock(0b00010000, &sircs_ext5);
    if (err == 1){ // timeout?
        sircs_ext5 = 0; // keine SIRCS-20 erweiterung empfangen
        return 0;
    } else if (err < 0) { // Fehler?
        sircs_ext5 = 0;
        return -5;
    }
    
    return 0;
}
// Ein Sircs Datenblock empfangen
// - sircs_bitmask definiert Position des MSB und damit auch die Blocklnge
// - emppfangene Daten werden per reference in sircs_data geschrieben
int8_t SIRCS_ReceiveBlock(uint8_t sircs_bitmask, uint8_t *sircs_data){
    uint8_t sircs_bitmask_loop;

    *sircs_data = 0;
    sircs_bitmask_loop = sircs_bitmask;
    do{
        // max 1200us auf Flanke Warten
        cnt_100us = 0;
        fIrPort_FF();
        while((cnt_100us < 12) && ((IrRcvPort_FF & IrRcv_Portbit) == 0)){
            fIrPort_FF();
        }
        if (cnt_100us >= 12)    // Timer out? -> Returncode 1
            return 1;

        // Flanke erkannt, 900us warten
        cnt_100us = 0;
        while(cnt_100us < 9){
            fIrPort_FF();
            if((IrRcvPort_FF & IrRcv_Portbit) != 0)
                return -1; // Fehlflanke erkannt -> Mit fehler zurck
        }

        // Bit auswerten
        *sircs_data >>= 1;
        if ((IrRcv_Port & IrRcv_Portbit) == 0)
            *sircs_data |= sircs_bitmask;
        sircs_bitmask_loop >>= 1;

    } while (sircs_bitmask_loop != 0);

    return 0;
}

// Erkennung fallende Flanke auf IR-Port
void fIrPort_FF(void){
    static uint8_t IrRcvPort_old;
    uint8_t temp = IrRcv_Port;

    IrRcvPort_FF = (temp ^ IrRcvPort_old) & IrRcvPort_old;
    IrRcvPort_old = temp;
    return;
}

void SircsToLCD(void){
    ClearLCD();
    SetDDRamAddrLCD(0x00);
    WriteStringLCD("cmd:0x");
    WriteHexLCD(sircs_code);
    itoa(tmpstring, sircs_cnt, 10);
    WriteStringLCD(" n:");
    WriteStringLCD(tmpstring);
    SetDDRamAddrLCD(0x40);
    WriteStringLCD("dev:0x");
    WriteHexLCD(sircs_address);
    if (sircs_ext3 != 0){
        WriteCharLCD(' ');
        WriteHexLCD(sircs_ext3);
    }
    else
        WriteStringLCD("   ");
    if (sircs_ext5 != 0){
        WriteCharLCD(' ');
        WriteHexLCD(sircs_ext5);
    }
    else
        WriteStringLCD("   ");


}

void ErrorToLcd(int8_t err){
    ClearLCD();
    itoa(tmpstring, err, 16);
    SetDDRamAddrLCD(0x40);
    WriteStringLCD("error: 0x");
    WriteStringLCD(tmpstring);
}

void delay1usLCD(void){
    __delay_us(1);
    return;
}
void delay5msLCD(void){
    __delay_ms(5);
    return;
}
void delay45msLCD(void){
    __delay_ms(45);
    return;
}
